% CircAdaptMain
% Theo Arts, Tammo Delhaas, Joost Lumens
% Version CircAdapt November, 2008 (TriSeg paper)
% Theo Arts, Maastricht University
% Email: t.arts@bf.unimaas.nl
% Joost Lumens, Maastricht University
% November, 2008. Email: j.lumens@fys.unimaas.nl
%
% CircAdapt is a lumped model of the dynamic circulation. Size of heart
% and bloodvessels adapt to load, as generated by the model itself.
% All variables and parameter are stored in the structure Par with file
% name 'Par'. Hemodynamic data are displayed graphically
% Map of structure 'Par' obtained by execution of 'mapstructure(Par)'
% Used functions: CircNew, CircAdap, CircAdaptDisplay

clear
global Par;

addpath(pwd);% add current directory to path

if exist('Par.mat'); % if Par.mat file is available
    FileName='Par';
    PathName=[pwd,'\'];
    ShowMenu=1;
    while ShowMenu;
        c2=input('[N]ew, [R]eference, [L]oad, [C]ontinue <Enter>: ','s');
        a=[c2,'c']; c=a(1); % convert <Enter> to 'c'
        switch lower(c)
            case('l'); % load file with Par-structure
                [FileName,PathName] = uigetfile('*.mat','Choose file: ');
                load([PathName,FileName]);
                c='c';ShowMenu=false;
            case('r') % load reference file with Par-structure
                load('ParRef');
                c='c';ShowMenu=false;
            case('n'); % new Par-structure from scratch
                CircNew;
                c='c';ShowMenu=false;
            case('c'); % continuation of simulation
                load([PathName,FileName]);
                c='c';ShowMenu=false;
            otherwise ShowMenu=true;
        end
    end
else
    if exist('ParRef.mat');
        c2=input('[N]ew, [R]eference, [L]oad,: ','s');
        a=[c2,'c']; c=a(1); % convert <Enter> to 'c'
        switch lower(c)
            case('l'); % load file with Par-structure
                [FileName,PathName] = uigetfile('*.mat','Choose file: ');
                load([PathName,FileName]);
                c='c';ShowMenu=false;
            case('r')
                load('ParRef');
                c='c';
            otherwise
                CircNew;
                c='c';
        end
    else
        Par=CircNew; % Parameter initialization, some remodeling rules inclusive
        % Generates parameter structure Par and initial conditions of the variables Par
        c='c';
    end
end

%menu items
MenuString={'Normal','Rest','Exercise'};
Par.Specific.MenuNr=1; % Normal contraction default
G=Par.General;
G.DtSimulation=1.5*G.tCycle; % standard duration of simulation

%XXXX Menu for changing hemodynamic variables and adaptation condition
OK=1;
AdaptStr=MenuString{Par.Specific.MenuNr};
while OK;
    disp(' ');
    disp(['[P]ressure                  (kPa): ',num2str(G.p0/1e3)]);
    disp(['[F]low                     (ml/s): ',num2str(G.q0*1e6)]);
    disp(['cycle [T]ime                 (ms): ',num2str(G.tCycle*1e3)]);
    disp(['[D]uration simulation         (s): ',num2str(G.DtSimulation)]);
    disp(['Adapt [R]est,[E]xercise          : ',AdaptStr]);
    disp(['<Enter> = Continue']);
    c1=input('Choose Letter <Enter>: ','s');
    switch lower(c1);
        case 'p'
            G.p0=input('Mean Arterial Pressure (kPa): ')*1e3;
        case 'f'
            G.q0=input('Systemic Flow (ml/s): ')/1e6;
        case 't'
            G.tCycle=input('Cycle Time (ms): ')*1e-3;
        case 'd'
            G.DtSimulation=input('Duration of simulation (s): ');
        case 'r'
            Par.Specific.MenuNr=2; %rest condition
            AdaptStr=MenuString{Par.Specific.MenuNr};
            G.DtSimulation=50*G.tCycle;
        case 'e'
            Par.Specific.MenuNr=3; %exercise condition
            AdaptStr=MenuString{Par.Specific.MenuNr};
            G.DtSimulation=100*G.tCycle;
        otherwise
            OK=0;
    end
end

% === Solves SVar for problem defined in parameter structure 'Par'
Par.General=G;
CircAdapt; %generate solution

%=== Saving State Variables and Model Parameters
save Par Par; %save compact solution as structure Par in file 'Par'
disp('Differential equation has been solved');

CircAdaptDisplay; % graphical display
% Structure Par has been extended with solution for all variables as a
% function of time
