function CircNew(varargin);
% function Par= CircNew(varargin);
% Theo Arts, Maastricht University
% May, 2008, email: t.arts@bf.unimaas.nl
% Reference: Arts et al., Am J Physiol Heart Circ Physiol.
% 2005;288:1943-1954.
% Initialization of simulation of hemodynamics and cardiac mechanics
% Left atrium, mitral valve, left ventricle, aortic valve, arteries, venes,
% Right atrium, tricuspid valve, right ventricle, pulmonary arteries, lung
% venes
% Par is a global structure, containing parameters and variables
% with sufficient information to start the problem
% The structure of Par can be viewed by executing <MapStruct(Par)>
% If varargin is something: some properties are copied from Par
% needed functions: Timing, CavityMech, TwinBubble

global Par;

%=== Default values
qRest     = 85e-6 ; % Cardiac Output, mean systemic flow
pRest     = 12200 ; % mean systemic blood pressure
pDropPulm = 1500  ; % pressure drop in pulmonary circulation
pResp     = 0*3000; % Thoracic pressure variations
TimeFac   = 1     ; % Time scale for contraction pulse

%==== Copies available Par data in new Par-structure
if nargin== 1; %if INPUT==Par structure, this is used as initial Par
    Par= varargin{1};
    General= Par.General;
    % overwriting of defaults
    if isfield(General,'qRest'    ); qRest= General.qRest    ; end
    if isfield(General,'pRest'    ); qRest= General.pRest    ; end
    if isfield(General,'pDropPulm'); qRest= General.pDropPulm; end
    if isfield(General,'TimeFac'  ); qRest= General.TimeFac  ; end
else
    Par= []; General= []; % default
end
Par.t=0; Par.tDot=1;
%----

%==== Time scaling by Systemic flow for Body size 
tCycleRest   = 0.85*(qRest/85e-6)^(1/3); % HR-scaling: Dawson et al
Dt           = 0.0005*2^round(log(tCycleRest/0.2)/log(2)); % time discretization ODD-solution (s)
DtSimulation = 5.5*tCycleRest  ; % minimum duration of simulation (s), will be rounded to a complete cycle

%==== Default hemodynamics set to rest
tCycle = tCycleRest;
p0     = pRest;
q0     = qRest;
rhob   = 1050        ; % blood density
%-----------------------------

%==== Default Valve properties
Valve.rhob= rhob; % blood density
Valve.q   = 0   ; % initialial condition no flow (SVar)
Valve.qDot= 0   ; % SVarDot

%==== Default Tube properties
Tube.k               = 8.0  ; % stiffness exponent vessel wall fibers, determines ZWave+++++
Tube.Adapt.WallStress= 500e3; % Wall stress at peak (systolic) pressure (Pa)
Tube.Adapt.vFlowMean = 0.17 ; % Mean flow velocity [m/s], Dawson
Tube.Adapt.vImpact   = 3.0  ; % Velocity of possible whole body impact, blood shockwave
Tube.q0              = qRest; % estimated mean tube flow for adaptation
Tube.rhob            = rhob ; % blood density (kgm-3)
%=== default systemic arterial Tube
Tube.p0  = pRest;
Tube.Len = 9.1*qRest^(1/3) ; % Dawson scaling for aortic length
Tube     = TubeInit(Tube)  ; % Wall and lumen size adapted
%-----------------------------------------

%==== Default (LV-) Sarcomere mechanics
%==== Adaptation parameters for sarcomere behavior
Sarc.Adapt.LsBe      = 2.2 ; % end-diastolic sarcomere length with excercise
Sarc.Adapt.LsEe      = 1.75; % end-systolic sarcomere length with excercise
Sarc.Adapt.SfPasMax  = 10000;% maximum passive G= stress/Ls
%==== Default ventricular sarcomere
Sarc.ActivationDelay = 0    ; % time of depolarization set to zero (= p-wave)
Sarc.Ef              = 0.0  ;
Sarc.LsRef           = 2.00 ; % [um] reference sarcomere length for AmRef
Sarc.Ls0Pas          = 1.80 ; % [um] sarcomere length with zero passive stress
Sarc.dLsPas          = 0.60 ; % [um] LsiStress0Pas+dLsPas equals stress pole
Sarc.SfPas           = 2500 ; % [Pa] passive stress at Ef=0
Sarc.Lsi0Act         = 1.51 ; % Zero force sarcomere length
Sarc.LenSeriesElement= 0.04 ; % [um] isometric dLs of series elastic element
Sarc.SfAct           = 12E4 ; % [Pa] max isometric stress
Sarc.vMax            = 7.0*0.85/tCycleRest; % [um/s] shortening velocity scaled to HR(Body weight)
Sarc.TimeAct         = 0.42 * tCycleRest; % [ m/s] duration force pulse
% duration at Ls= Lsi0Act (s), (Arts, AJP, 1982)
Sarc.TR              = 0.25 ; % ratio rise time to TimeAct
Sarc.TD              = 0.25 ; % ratio decay time to TimeAct
Sarc.C               = 0.0  ; % [Ca++] concentration factor (SVar)
Sarc.CDot            = 0.0  ; % SVarDot
Sarc.CRest           = 0.02 ; % diastolic resting C-value (contractility)
Sarc.Lsi             = Sarc.Adapt.LsBe;
Sarc.LsiDot          = 0; %SVarDot

%==== Defining Cavity fields
Cavity= struct('Sarc',[],'AmRef',[],'AmDead',[],'VWall',[],'V',[],'VDot',[],'Vm',[],...
    'Xm',[],'Ym',[],'Am',[],'Cm',[],'T',[],'pTrans',[],'DTDAm',[],...
    'p',[],'A',[],'Z',[],'pIn',[],'qRemod',[],'Y',[],'YDot',[],...
    'Tau',[] );
Cavity.Sarc = Sarc;
%----------------------------

%======================= SPECIFIC PROPERTIES ============================= 

%==== Mean Valve Flows for initial adaptation
Par.ValveLVen= Valve; % default
Par.ValveLAv = Valve; % default
Par.ValveLArt= Valve; % default
Par.ValveRVen= Valve; % default
Par.ValveRAv = Valve; % default
Par.ValveRArt= Valve; % default
Par.ValveASD = Valve; % default
Par.ValveVSD = Valve; % default
Par.ValveDUCT= Valve; % default
%-----

%=== valve connections to tubes/chambers
Par.ValveLVen.Prox= 'TubeLVen'; Par.ValveLVen.Dist= 'La'; 
Par.ValveLAv.Prox = 'La'      ; Par.ValveLAv.Dist = 'Lv'; 
Par.ValveLArt.Prox= 'Lv'      ; Par.ValveLArt.Dist= 'TubeLArt'; 
Par.ValveRVen.Prox= 'TubeRVen'; Par.ValveRVen.Dist= 'Ra'; 
Par.ValveRAv.Prox = 'Ra'      ; Par.ValveRAv.Dist = 'Rv'; 
Par.ValveRArt.Prox= 'Rv'      ; Par.ValveRArt.Dist= 'TubeRArt'; 
Par.ValveDUCT.Prox= 'TubeLArt'; Par.ValveDUCT.Dist= 'TubeRArt'; 
Par.ValveVSD.Prox = 'Lv'      ; Par.ValveVSD.Dist = 'Rv'; 
Par.ValveASD.Prox = 'La'      ; Par.ValveASD.Dist = 'Ra'; 
%-----

%==== Setting independent flows through valves
Par.ValveRVen.q = qRest; % RA inflow
Par.ValveLVen.q = qRest; % LA inflow
Par.ValveASD.q  = 0;     % ASD flow left->right
Par.ValveVSD.q  = 0;     % VSD flow left->right
% calculating dependent flows
Par.ValveLAv.q = Par.ValveLVen.q - Par.ValveASD.q ;
Par.ValveRAv.q = Par.ValveRVen.q + Par.ValveASD.q ;
Par.ValveLArt.q= Par.ValveLAv.q  - Par.ValveVSD.q ;
Par.ValveRArt.q= Par.ValveRAv.q  + Par.ValveVSD.q ;
Par.ValveDUCT.q= Par.ValveLVen.q - Par.ValveRArt.q;

%==== TUBES
%==== Left Arteries= Aorta
Par.TubeLArt      = Tube               ; % Aorta as a tube
Par.TubeLArt.p0   = pRest              ; % estimated mean pressure
Par.TubeLArt.q0   = Par.ValveLArt.q    ; % flow to adapt to
Par.TubeLArt= TubeInit(Par.TubeLArt)   ; % Wall and lumen size adapted

%==== Left Venes (pulmonary)
Par.TubeLVen      = Tube               ; % Venes as a tube with open end to heart
Par.TubeLVen.k    = 10.0               ; % stiffer venular walls
AuxSf= Sarc.Adapt.SfPasMax/Sarc.SfAct; %ratio passive/active pressure
Par.TubeLVen.p0   = 0.4270*AuxSf*p0    ; % estimated mean venous pressure
Par.TubeLVen.q0   = Par.ValveLVen.q    ; % flow to adapt to
Par.TubeLVen.Len  = 0.5*Tube.Len       ; % pulmonary vessels are shorter
Par.TubeLVen= TubeInit(Par.TubeLVen)   ; % Wall and lumen size adapted

%==== Right Venes (systemic)
Par.TubeRVen      = Par.TubeLVen       ; % Venes as a tube with open end to heart
Par.TubeRVen.p0   = 0.2845*AuxSf*p0    ; % estimated mean venous pressure
Par.TubeRVen.q0   = Par.ValveRVen.q    ; % flow to adapt to
Par.TubeRVen.Len  = Tube.Len           ;
Par.TubeRVen = TubeInit(Par.TubeRVen)  ; % Wall and lumen size adapted

%==== Right Arteries (Pulmonary)
Par.TubeRArt      = Par.TubeLArt       ; % Pulmonary trunc as a tube
Par.TubeRArt.p0   = pDropPulm+Par.TubeRVen.p0; % estimated mean pressure
Par.TubeRArt.q0   = Par.ValveRArt.q    ; % flow to adapt to
Par.TubeRArt.Len  = 0.5*Tube.Len       ; % pulmonary vessels are shorter
Par.TubeRArt = TubeInit(Par.TubeRArt)  ; % Wall and lumen size adapted

%==== Peripheral resistance
Par.LRp.R= (Par.TubeLArt.p0-Par.TubeLVen.p0)/Par.TubeLArt.q0; %set to normal
Par.RRp.R= pDropPulm/Par.TubeRArt.q0; %pulmonary resistance

%==== VALVES%+++++++++++++++++++++++++++++++++
% setting isometric load to heart cavities
Par.ValveLVen.AOpen  = Par.TubeLVen.A0;     % pulmonary venous orifice area
Par.ValveLAv.AOpen   = 2.0*Par.TubeLArt.A0; % mitral valve area
Par.ValveLArt.AOpen  = Par.TubeLArt.A0;     % aortic valve area
Par.ValveRVen.AOpen  = Par.TubeRVen.A0;     % systemic venous orifice area
Par.ValveRAv.AOpen   = 2.0*Par.TubeRArt.A0; % tricuspid valve area
Par.ValveRArt.AOpen  = Par.TubeRArt.A0;     % pulmonary valve area
Par.ValveDUCT.AOpen  = Tube.A0; % Ductus area
Par.ValveVSD.AOpen   = Tube.A0; % VSD area
Par.ValveASD.AOpen   = Tube.A0; % ASD area

Str= {'LAv', 'LArt', 'RAv', 'RArt', 'DUCT', 'VSD', 'ASD'};
for i= 1:length(Str);
    s1= ['Par.Valve',Str{i}]; P1= eval(s1);
    P1.ALeak   = 1e-6*P1.AOpen; % setting small valve leaks
    P1.Len     = 0.9*sqrt(P1.AOpen); % inertial length of orifice
    eval([s1,'= P1;']);
end
Str= {'LVen', 'RVen'};
for i= 1:length(Str);
    s1= ['Par.Valve',Str{i}]; P1= eval(s1);
    P1.ALeak   = P1.AOpen; % setting valve leak equal to forward area
    P1.Len     = 0.9*sqrt(P1.AOpen); % inertial length of orifice
    eval([s1,'= P1;']);
end


%==== Ductus forward (left->right) closure
Par.ValveDUCT.AOpen  = Par.ValveDUCT.ALeak; % Ductus cross-section
Par.ValveVSD.AOpen   = Par.ValveVSD.ALeak ; % VSD cross-section
Par.ValveASD.AOpen   = Par.ValveASD.ALeak ; % ASD cross-section


%==== CAVITIES
%==== Left Atrial cavity, estimate for excercise
Par.La = Cavity; % default

VStrokeAdapt= Par.ValveLVen.q*tCycle;

pMax   = 0.2733*Par.TubeLVen.pMax   ; % maximum La pressure for adaptation
Par.La = InitCavity(Par.La,0.6716*VStrokeAdapt,pMax)  ; % Adaptation of wall and cavity size
%==== Atrium specific sarcomere behavior
Par.La.Sarc.vMax = 2.0*Sarc.vMax; % faster contraction rate of atria
Par.La.Sarc.TR   = 0.5     ; % longer rise time relative to pulse duration
Par.La.Sarc.TD   = 0.5     ; % longer decay time relative to pulse duration
Par.La.Sarc.SfAct= 0.7*Sarc.SfAct;
Par.La.Sarc.Adapt.SfPasMax= Sarc.Adapt.SfPasMax*10.0;  % stiffer passive behavior
Par.La.Sarc.Adapt.LsEe    = Par.La.Sarc.Adapt.LsEe-0.1;% further contraction
Par.La.AmDead= Par.ValveLAv.AOpen+Par.ValveLVen.AOpen;

%==== Right atrium
Par.Ra = Par.La; % default
Par.Ra.VWall= 0.4546*Par.La.VWall;
Par.Ra.AmDead= Par.ValveRAv.AOpen+Par.ValveRVen.AOpen;

%==== Left Ventricular cavity, inner shell
Par.Lv = Cavity;
pMax   = (Par.TubeLArt.p0-Par.TubeRArt.p0); % maximum LV pressure for adaptation
Par.Lv = InitCavity(Par.Lv,VStrokeAdapt,pMax); % Adaptation of wall and cavity size

%==== Right Ventricular cavity, outer shell
Par.Rv = Par.Lv;
pMax   = Par.TubeRArt.p0; % maximum RV pressure for adaptation
Par.Rv = InitCavity(Par.Rv,VStrokeAdapt,pMax); % Adaptation of wall and cavity size

%==============================================
%=== Ventricular Twin-cavity

Par.Sv= Par.Lv; %copy Lv to Sv
% Par.Sv.Sarc.Adapt.SfPasMax= Sarc.Adapt.SfPasMax*0.5;  % compliant septum
Par.Sv.Sarc.Adapt.SfPasMax= Sarc.Adapt.SfPasMax*0.8;  % compliant septum+++++++++++
Par.Rv.Sarc.Adapt.SfPasMax= Sarc.Adapt.SfPasMax*2.0 ;  % stif RV wall

Par.Lv.AmDead= Par.ValveLAv.AOpen+Par.ValveLArt.AOpen;
Par.Rv.AmDead= Par.ValveRAv.AOpen+Par.ValveRArt.AOpen;
Par.Sv.AmDead= 0;

% cavity volumes
VL=Par.Lv.V; VR=Par.Rv.V;
Par.Lv.V=0.8314*VL; %volumes
Par.Rv.V=0.9978*VR;

AL=Par.Lv.AmRef;  AR=Par.Rv.AmRef; %reference wall area
Par.Lv.AmRef=0.7591*AL;
Par.Sv.AmRef=0.3178*AL;
Par.Rv.AmRef=0.9532*AR;
hL=Par.Lv.VWall/AL; hR= Par.Rv.VWall/AR; % wall thicknesses
hL=hL+0.5*hR; hS=hL;
Par.Lv.VWall=Par.Lv.AmRef*hL*1.4549;
Par.Sv.VWall=Par.Sv.AmRef*hS*1.4216;
Par.Rv.VWall=Par.Rv.AmRef*hR*1.8648;

Par.Sv.Y  = sqrt(0.9221*Par.Sv.AmRef/pi); Par.Sv.YDot=0;
Par.Sv.V  = 0.0556*(Par.Lv.V+0.5*Par.Lv.VWall);
Par.Sv.Tau= 0.01; % setting time constant of correction for Vm, Ym in septum

%=== Setting unloaded sarcomere length

Par.La         = CavityMech(Par.La)  ; % Estimate of initial sarcomere length and cavity volume
Par.La.Sarc.Lsi= Par.La.Sarc.Ls-Par.La.Sarc.LenSeriesElement;

Par.Ra         = CavityMech(Par.Ra)  ; % Estimate of initial sarcomere length and cavity volume
Par.Ra.Sarc.Lsi= Par.Ra.Sarc.Ls-Par.Ra.Sarc.LenSeriesElement;


[Par.Lv,Par.Sv,Par.Rv]=TwinBubble(Par.Lv,Par.Sv,Par.Rv);
Par.Sv.V=Par.Sv.Vm;
Par.Sv.Y=Par.Sv.Ym;

% % Initialization unloaded sarcomere length
Par.Lv.Sarc.Lsi= Par.Lv.Sarc.Ls-Par.Lv.Sarc.LenSeriesElement;
Par.Sv.Sarc.Lsi= Par.Sv.Sarc.Ls-Par.Sv.Sarc.LenSeriesElement;
Par.Rv.Sarc.Lsi= Par.Rv.Sarc.Ls-Par.Rv.Sarc.LenSeriesElement;
%---

%===Pericardium
V=Par.Lv.V+Par.Rv.V+Par.Lv.VWall+Par.Sv.VWall+Par.Rv.VWall...
    +Par.La.V+Par.La.VWall+Par.Ra.V+Par.Ra.VWall;
Peri.VRef  = 1.4838*mean(V);
Peri.pAdapt= 500;
Peri.k     = 10;
Par.Peri   = Peri;
%-----

%==============================================


%=========== SETTING INITIAL STATE

% General data in Par.General
General.qRest        = qRest;
General.pRest        = pRest;
General.pDropPulm    = pDropPulm;
General.pResp        = pResp;
General.tCycleRest   = tCycleRest;
General.Dt           = Dt;
General.DtSimulation = DtSimulation;
General.tCycle       = tCycle;
General.p0           = p0;
General.q0           = q0;
General.rhob         = rhob;
General.TimeFac      = TimeFac;
Par.General=General; %contains general data

%sequence of activation
Timing; %timing function

% Par.Specific information
Par.Specific.MenuNr= 1; % normal beats, no daptation
if ~isfield(Par.Specific,'Menu');
    Par.Specific.Menu= [];
end
for i= 1:3
    Par.Specific.Menu(i).NameAdapt= 'Adapt0'; %no adaptation
end
% Setting adaptation function strings
Par.Specific.Menu(2).NameAdapt= 'AdaptRest';
Par.Specific.Menu(3).NameAdapt= 'AdaptExc' ;

Par=WriteSVarFiles(Par); % Writes Script Files for conversions

return

% ============== Auxiliary functions ================ 

function Tube= TubeInit(Tube);
Tube.V    = 0; % locate volume (SVar)
Tube.VDot = 0; % Tube volume Dot (SVarDot)
Tube.pMax = Tube.p0 + 0.5*Tube.rhob*Tube.Adapt.vImpact^2; %temporary estimate
Tube      = TubeGeometry(Tube);
Tube.pMax = Tube.p0 + 2.0*abs(Tube.Z.*Tube.A0*Tube.Adapt.vImpact);
Tube      = TubeGeometry(Tube)        ; % Wall and lumen size adapted
Tube.V    = Tube.A0*Tube.Len; % Tube volume (SVar)
return

function Cav = InitCavity(Cav,VStroke,pMax);
% estimate of begin-ejection geometry
SfMax= 0.4*Cav.Sarc.SfAct; % effective systolic stress
Ef=log(Cav.Sarc.Adapt.LsBe/Cav.Sarc.Adapt.LsEe); % systolic strain
Aux=exp(Ef/2);
Vm= VStroke/(Aux^3-Aux^-3); %mid ejection midwall enclosed volume
pDSf=pMax/SfMax; a=exp(3*pDSf); z=(a-1)/(a+1);
VWall= 2 * z * Vm; % estimate of wall volume
Lambda = Cav.Sarc.Adapt.LsBe/Cav.Sarc.LsRef;
Cav.AmRef= 4*pi*(Vm*3/(4*pi))^(2/3) * Lambda^2; %estimate of midwall area
Cav.VWall= VWall;
Cav.V= Vm - VWall/2 + VStroke/2; % estimate of end-diastolic volume
Cav.VDot=0;
return

function Tube=TubeGeometry(Tube);
% function Tube=TubeGeometry(Tube);
% Theo Arts, Maastricht University
% July 31, 2006, email: t.arts@bf.unimaas.nl
% Vessel diameter adapts to flow, wall thickness to maximum pressure
    
A0   = Tube.q0/Tube.Adapt.vFlowMean; % A~flow-> constant velocity, Dawson et al
pMax = abs(Tube.pMax);
SfMax= Tube.Adapt.WallStress;
p0   = Tube.p0;
k    = Tube.k;

Tube.AWall= 3*A0/((SfMax/pMax)*(p0/pMax)^(3/(k-3))-1);
Tube.A0   = A0;

Tube.V(end)= A0*Tube.Len;
Tube= TubeDynamics(Tube); % add values of A, p and Z

return


%Tested OK:SfMax= pMax^(k/(k-3))*(1+3*A0/Tube.AWall)/p0^(3/(k-3))

%EQUATIONS likely to be about OK
%ex^3= (1+3AMax/AWall)/(1+3A0/AWall);
%s0= (1+3A0/AWall)*p0
%sMax= s0*ex^k
%sMax= (1+3AMax/AWall).pMax
%known: k,A0,pMax,p0,sMax
%to be solved: AWall

