function Sheet= MultiPatch(Sheet);
% function Sheet= MultiPatch(Sheet);
% A 2D- Sheet is subdivided in nPatch patches Sheet.Sarc. Sarcomere
% properties are stored in Sheet.Sarc.Sarc
% Input: total Sheet area and curvature
% From dimensions, areas Am of patches are calculated, using equilibrium of
% wall tension
% Output: Mechanics of patches and in total Sheet tension and pressure
% Structure Sheet
% Sheet.
%       Sarc: [1x1 struct]
%      AmRef: 0.0120
%      VWall: 1.0658e-004
%          V: [851x1 double]
%       VDot: [851x1 double]
%         Vm: [851x1 double]
%         Xm: [851x1 double]
%         Ym: [851x1 double]
%         Am: [851x1 double]: In
%         Cm: [851x1 double]: In
%          T: [851x1 double]
%     pTrans: [851x1 double]
%      DTDAm: [851x1 double]
%          p: [851x1 double]
%          A: [851x1 double]
%          Z: 0
%        pIn: [851x1 double]
%     qRemod: [851x1 double]
%          Y: []
%       YDot: []
%        Tau: []
% Structure Sarc for 3 patches:
% Sheet.Sarc. Info about separate patches
%     VWall: [3.5528e-005 3.5528e-005 3.5528e-005]
%     AmRef: [0.0040 0.0040 0.0040]
%        Cm: [851x3 double]
%        Am: [851x3 double]
%      Sarc: [1x1 struct]

% 
% collecting input
% Sheet is divided in number of patches. Properties of patches stored in
% Sheet.Sarc. Related sarcomeres in Sheet.Sarc.Sarc.
Cm    = Sheet.Cm; %curvature f(t)
Am    = Sheet.Am; %area f(t)
Patch = Sheet.Sarc ; % 'Patch(1)' represents array of patches
VWall = Patch.VWall; % wall volume row (p)
AmRef = Patch.AmRef; % AmRef row
VWallTot= sum(VWall,2);
nt    = size(Am,1);
nPatch= size(VWall,2); % number of patches (p)
Sarc  = Patch.Sarc;    % Array[T,p] of Sarcomere properties
Lsi   = Sarc.Lsi;
LsRef = Sarc.LsRef;

% filling Patches with first estimate of Am with Ls=Lsi
AmLsi = Lsi.^2 * sparse(diag(AmRef./LsRef.^2)); %Am for Ls=Lsi
Patch.Am = AmLsi;
Patch.Cm = repmat(Cm,[1,nPatch]);
Patch    = SheetMech(Patch);
T        = Patch.T;
DTDA     = Patch.DTDAm;

% solve areas of Patches on balance of Tension
% Equations: sum(Patch.Am)=Sheet.Am
% All Patch.T are equal
% All Patches: T=T0+DTDA*DA
Row1 = ones(1,nPatch);
DATot= Am-sum(AmLsi,2); % (T=Time) column vector
ADT  = sum(1./DTDA,2); % (T) column vector
T0   = mean(T.*AmLsi,2)./mean(AmLsi,2); % (T) column vector
DA0  = sum( (T0*Row1-T)./DTDA ,2); % (T) column vector
DT0  = (DATot-DA0)./ADT; % (T) column vector
DT   = (T0+DT0)*Row1-T; %(Tp) matrix
DA   = DT./DTDA; %(Tp) matrix
A2   = AmLsi+DA; %(Tp) matrix

Patch.Am = A2; % new estimate of Am (Tp)-matrix
Patch    = SheetMech(Patch);

Sheet.Sarc= Patch; % collection of patches
T = Patch.T*VWall'/VWallTot; % Tension
Sheet.pTrans= 2 * T .* Cm  ; % transmural pressure
Sheet.T     = T;
Sheet.VWall = VWallTot;
Sheet.DTDAm = 1./ADT;

return
