function Sarc= SarcMech(Sarc);
%function Sarc= SarcMech(Sarc);
% Theo Arts, Maastricht University
% Email: t.arts@bf.unimaas.nl
% Joost Lumens, Maastricht University
% November, 2008. Email: j.lumens@fys.unimaas.nl
%Sarcomere mechanics, G= Stress/sarcomere length= function of time and sarcomere length
%contractility and other parameters may be varied by varying Sarc-values
% matrix proof: vertical->t; horizontal -> different sarcomeres
% structure Sarc. for wall with 3 patches
%      ActivationDelay: [3x3 double]
%                   Ef: [851x3 double]
%                LsRef: [1.9973 1.9973 1.9973]
%               Ls0Pas: [1.8000 1.8000 1.8000]
%               dLsPas: [0.6000 0.6000 0.6000]
%                SfPas: [3.9913e+003 3.9913e+003 3.9913e+003]
%              Lsi0Act: [1.5100 1.5100 1.5100]
%     LenSeriesElement: [0.0400 0.0400 0.0400]
%                SfAct: [120000 120000 120000]
%                 vMax: [7 7 7]
%              TimeAct: [0.4250 0.4250 0.4250]
%                   TR: [0.2500 0.2500 0.2500]
%                   TD: [0.2500 0.2500 0.2500]
%                    C: [851x3 double]
%                 CDot: [851x3 double]
%                CRest: [0.0200 0.0200 0.0200]
%                  Lsi: [851x3 double]
%               LsiDot: [851x3 double]
%                Adapt: [1x1 struct]

global Par; % general time

%==== Input variables
t       = Par.t;
Ef      = Sarc.Ef ;
[nt,nr] = size(Ef); % nt: number of times; nr: number of sarcomeres
tc      = Tc(t,Sarc.ActivationDelay);
Lsi     = Sarc.Lsi;
C       = Sarc.C  ;
LenSeriesElement= repmat(Sarc.LenSeriesElement,[nt,1]);
TR      = repmat(Sarc.TR     ,[nt,1]); %TR atrial muscle > ventricular muscle
TD      = repmat(Sarc.TD     ,[nt,1]);
TimeAct = repmat(Sarc.TimeAct,[nt,1]); %time scale of contraction pulse
Ls0     = repmat(Sarc.Lsi0Act,[nt,1]); %zero active stress sarcomere length
Ls0Pas  = repmat(Sarc.Ls0Pas ,[nt,1]);
dLsPas  = repmat(Sarc.dLsPas ,[nt,1]);
SfPas   = repmat(Sarc.SfPas  ,[nt,1]);
CRest   = repmat(Sarc.CRest  ,[nt,1]); %Resting C-value (Ca++  contractility)

% series elasticity and sarcomere shortening velocity
Ls         = exp(Ef) * sparse(diag(Sarc.LsRef));
Sarc.Ls    = Ls; 

% constants related to timing are mainly based on experimental findings
L  = max(Lsi-Ls0,0.0001); % normalized sarc length for active contraction
tA = TimeAct.*(0.74+0.3*L); % activation time lengthens with sarcomere length
tR = 0.45*TR.*TimeAct     ; % rise time
tD = 0.30*TD.*TimeAct     ; % decay time (default 0.3)
T  = tc./tR;
x=min(8,max(0,T));
ft1= x.^3 .* exp(-x) .* (8-x).^2 * 0.020 ./ tR;

%         rise of contraction, 'amount of Ca++ release'

%Integral T^n exp(-T) = Gamma(n+1) = n!
FL= tanh(4.0*L.^2); % regulates increase of contractility with Ls
x=(tc-tA)./tD; y= min(3,max(-3,x));
tanhx = 0.5 + 0.25*y - 9.2593e-3 * y.^3;
% Time confined approximation of 1/(1-e^x) function
Sarc.CDot= FL.*ft1 - (C-CRest).*tanhx./tD; % 1st order rise and decay of [Ca++]
% Approximation of Sarc.CDot= FL.*ft1 - C./((1+exp(-x))*tD)
% representing 1st order rise and decay of [Ca++]
SfIso    =  C.* L * sparse(diag(Sarc.SfAct));

%=== Passive stress ans stiffness characteristics
kS    = Ls0Pas./dLsPas; %Sarcomere stiffness (Titin?)
aS    = 4.0*kS.^2;
a0S   = 0.1; % Small linear sarcomere stiffness (can be <0)

S    = double(Ls>Ls0Pas); % positive sarcomere stretch
EfS0 = log(Ls./Ls0Pas)  ; % sarcomere strain, may be<0 (pushing)
EfS  = S .* EfS0;

kM= 10*kS; % Passive stiffness of matrix
y        =   aS.*EfS.^2 + a0S*EfS0 +     exp(Ef.*kM-0.2*kM)  ;
DyDEf    = 2*aS.*EfS    + a0S      + kM.*exp(Ef.*kM-0.2*kM)  ;
SfPasT   = SfPas.*y    ;
DSfPasDEf= SfPas.*DyDEf;

%=== Stress/Ls and stiffness, collected for output
Sarc.SfPasT = SfPasT;% passive stress
LNormSe     = max( -0.01,(Ls-Lsi)./LenSeriesElement);
%       normalized Series Elastic Element length
NoBuckle    = tanh(10*C + max(0,1e-4*SfPasT.^2));% NoBuckle: Lsi follows Ls
Sarc.LsiDot = max(LNormSe-1,NoBuckle.*(LNormSe-1)) * sparse(diag(Sarc.vMax));
Sarc.Sf     = SfPasT + SfIso.*LNormSe;
Sarc.DSfDEf = DSfPasDEf+SfIso.*Ls./LenSeriesElement; % estimate of sarcomere stiffness

return

function tc=Tc(t,ttrig); % 'matrix proof'
nt=size(t,1);
[nc,nr]=size(ttrig);
ttrigT=ttrig';
A=double(repmat(t,[1,nr*nc])>repmat(ttrigT(:)',[nt,1]));
B=reshape( sum(reshape(A,[nt*nr,nc]),2), [nt,nr]);
b=[zeros(1,nr)-10;ttrig];
tc=repmat(t,[1,nr])-b(B+1);
return

