function Timing;
% function Timing;
% Theo Arts, Maastricht University
% Email: t.arts@bf.unimaas.nl
% Joost Lumens, Maastricht University
% November, 2008. Email: j.lumens@fys.unimaas.nl
% Renders arrays of activation times (Sarc.ActivationDelay) per sarcomere
% Start of simulation: Par.t(end)
% Duration of simulation: Par.General.DtSimulation.
% To be executed at the beginning of a simulation CircAdapt(Par)
% Allows variation of spread of activation and irregular heart beat

global Par

% Reading timing parameters
G        = Par.General;
tCycleRef= G.tCycleRest ; % Reference tCycle for body size scaling
tCycle   = G.tCycle     ; % current cycle time
TimeFac  = G.TimeFac    ; % scaling of contraction time

% TimeAct [s] duration of contraction for Ls==LsStress0Act (s)
Par.La.Sarc.TimeAct= 0.20*(tCycleRef/0.85)*TimeFac;
Par.Ra.Sarc.TimeAct= Par.La.Sarc.TimeAct;
Par.Lv.Sarc.TimeAct= (0.10*tCycleRef +0.40*tCycle)*TimeFac;
Par.Sv.Sarc.TimeAct= Par.Lv.Sarc.TimeAct;
Par.Rv.Sarc.TimeAct= Par.Lv.Sarc.TimeAct;

% Delay times electrical activation
tRa2La= 0.02*tCycleRef/0.85 *TimeFac;
tRa2Rv= Par.Ra.Sarc.TimeAct -0.00*tCycle*TimeFac - 0.00*tCycleRef*TimeFac/0.85; %TauAv
tRv2Lv= -0.00;       % pacing delay LBBB, negative: prepacing
tRv2Sv= 0.5*tRv2Lv;  % pacing delay LBBB, negative: prepacing
tRa2Ra= tCycle;      % cycle time
% tRa2Ra= tRa2Ra*(1+0.3*(rand-0.5)); % irregular HR
DtLv   = 0.10;       % dispersion Lv activation time

% Time interval of simulation
Par.t(end)=0;
tStart= Par.t(end);
tEnd  = tStart+G.DtSimulation;
Par.General.tEnd= tEnd;

% initialization
if tStart==0; % reset all timing variables
    tRa= 0;
else
    tRa= FindLastBefore( ActivationDelay(Par.Ra), tStart );
end

% ActivationDelay: array of activation times
tLa=[]; tLv=[]; tSv=[]; tRv=[];
t=tRa;
while t<tEnd;
    tLa = [tLa; tRa(end) + tRa2La];
    tRv = [tRv; tRa(end) + tRa2Rv];
    tSv = [tSv; tRv(end) + tRv2Sv];
    tLv = [tLv; tRv(end) + tRv2Lv];
    tRa = [tRa; tRa(end) + tRa2Ra];
    t= tRa(end); % start new cycle
end

% Reset by setting Par.t(end)=0;
if tStart==0;
    tRa=[tRa(1)-tCycle;tRa];
    tLa=[tLa(1)-tCycle;tLa];
    tLv=[tLv(1)-tCycle;tLv];
    tSv=[tSv(1)-tCycle;tSv];
    tRv=[tRv(1)-tCycle;tRv];
else
    % Sarcomeres may be active from previous beat
    tLa= [FindLastBefore(ActivationDelay(Par.La),tStart);tLa];
    tLv= [FindLastBefore(ActivationDelay(Par.Lv),tStart);tLv];
    tSv= [FindLastBefore(ActivationDelay(Par.Sv),tStart);tSv];
    tRv= [FindLastBefore(ActivationDelay(Par.Rv),tStart);tRv];
end

Par.Ra= SheetAct(Par.Ra,{tRa});
Par.La= SheetAct(Par.La,{tLa});
Par.Lv= SheetAct(Par.Lv,{tLv,DtLv});% spread of activation over patches
Par.Sv= SheetAct(Par.Sv,{tSv});
Par.Rv= SheetAct(Par.Rv,{tRv});

return

%=========== Auxilary functions =========================================

function tLast= FindLastBefore(tAll,tRef);
    Rg= find(tAll<tRef);
    if length(Rg)==0; Rg=1; end
    tLast= tAll(Rg(end));
return


function tAct= ActivationDelay(Sheet);
if isfield(Sheet.Sarc,'Sarc')
    tAct= Sheet.Sarc.Sarc.ActivationDelay; %first patch
else
    tAct=Sheet.Sarc.ActivationDelay;
end
return


function Sheet= SheetAct(Sheet,t0Dt);
% Sets activation trigger times for sarcomeres in sheets or subsheets
% t0Dt={t0,Dt}
t0= t0Dt{1};
if isfield(Sheet.Sarc,'Am')
    Patch= Sheet.Sarc;
    nPatch= size(Patch.Am,2);
    Dt= ( (1:nPatch)-0.5*(nPatch+1) )/nPatch * t0Dt{2};
    nt= size(t0,1); np= size(Dt,2);
    Patch.Sarc.ActivationDelay= repmat(t0,[1,np])+repmat(Dt,[nt,1]);
    Sheet.Sarc= Patch;
else
    Sheet.Sarc.ActivationDelay= t0;
end
    