function [Lv,Sv,Rv]=TwinBubble(Lv,Sv,Rv);
% function [Lv,Sv,Rv]=TwinBubble(Lv,Sv,Rv);
% [Lv,Sv,Rv] represent left, septal and right wall segments
% Lv and Rv contain also cavity hemodynamics
% Sv contains also Left-Right interaction data
%
% input: Lv and Rv volumes, sarcomeres, wall sizes
% output: Segment geometry, wall area and curvature,
%   Sarcomere strain, sarcomere stress, Wall tension, pressures
%   time derivatives of state variables: LsiDot, YmDot, VmDot
% Joost Lumens, Maastricht University
% November, 2008. Email: j.lumens@fys.unimaas.nl
% Theo Arts, Maastricht University
% Email: t.arts@bf.unimaas.nl


Y= Sv.Y; V= Sv.V; % state variable, time dependent
Lv.Ym= Y; Sv.Ym= Y; Rv.Ym= Y; Sv.Vm= V;
% Lv.V, Rv.V, Sv.Vm -> Lv.Vm and Rv.Vm
Lv.Vm= Sv.Vm-Lv.V-0.5*(Lv.VWall+Sv.VWall);
Rv.Vm= Sv.Vm+Rv.V+0.5*(Sv.VWall+Rv.VWall);
Lv=VYm2ACXm(Lv) ; Sv=VYm2ACXm(Sv) ; Rv=VYm2ACXm(Rv);
Lv=SheetMech(Lv); Sv=SheetMech(Sv); Rv=SheetMech(Rv);

%determine stiffness matrix with numerical differentiation DV, DY
LRef= mean(Y); %averaged over time, scalar
DV= 0.04*LRef.^3; DY=0.02*LRef; % differentials
StrSheet= {'AmRef','VWall','Vm','Ym','DTDAm','Am','AmDead','T'};
LSRv=StructMerge(repmat([Lv,Sv,Rv],[1,3]),StrSheet);
A0            = LSRv.Am;
T0            = LSRv.T;
LSRv.Vm(:,4:6)= LSRv.Vm(:,1:3)+DV;
LSRv.Ym(:,7:9)= LSRv.Ym(:,1:3)+DY;

% structureLSRv:
% LSRv.Sarc(1).Var(9)
% LSRv.Var(9)
LSRv   = VYm2ACXm(LSRv) ; % 3D geometry to Sheet
% Fast linear version of SheetMech:
LSRv.T = T0+(LSRv.Am-A0).*LSRv.DTDAm; % Lineair estimate of Tension only
LSRv   = T2TxTy(LSRv);
%=== Equilibrium of tension at node

Tx_Twd=LSRv.Tx;
Ty_Twd=LSRv.Ty;

% summing Tx and Ty over Lv,Sv and Rv wall
M= sparse([1,1,1,0,0,0,0,0,0;...
           0,0,0,1,1,1,0,0,0;...
           0,0,0,0,0,0,1,1,1]');
Tx_Td= Tx_Twd*M;
Ty_Td= Ty_Twd*M;

Tx0= Tx_Td(:,1);
Ty0= Ty_Td(:,1);
D11= (Tx_Td(:,2)-Tx0); D21= (Ty_Td(:,2)-Ty0);
D12= (Tx_Td(:,3)-Tx0); D22= (Ty_Td(:,3)-Ty0);
% Jacobian matrix and matrix inversion
Det= max(D11.*D22-D12.*D21,0.1*(abs(D11.*D22)+abs(D12.*D21)));
FacV= (-D22.*Tx0+D12.*Ty0)./Det;
FacY= (+D21.*Tx0-D11.*Ty0)./Det;

% Substitute calculated TwinBubble geometry
DVm=repmat(FacV*DV,[1,3]);
DYm=repmat(FacY*DY,[1,3]);
LSRv.Vm(:,1:3)= LSRv.Vm(:,1:3)+DVm;
LSRv.Ym(:,1:3)= LSRv.Ym(:,1:3)+DYm;
LSRv= VYm2ACXm(LSRv) ;

% Copy geometry to wall segments
FieldsOut={'Vm','Ym','Am','Cm'};
nField= length(FieldsOut);
for iField=1:nField;
    Field= FieldsOut{iField};
    MTwd= LSRv.(Field);
    Lv.(Field)=MTwd(:,1);
    Sv.(Field)=MTwd(:,2);
    Rv.(Field)=MTwd(:,3);
end
% Get stress, tension and pressures and Sarc.CDot/LsiDot
Lv=SheetMech(Lv); Sv=SheetMech(Sv); Rv=SheetMech(Rv);

% transmural pressures -> cavity pressures
Lv.p= -Lv.pTrans;
Rv.p= +Rv.pTrans;
Sv.VDot= FacV*DV/Sv.Tau;
Sv.YDot= FacY*DY/Sv.Tau;

% Calculate hemodynamics of cavities
Lv=CavityHemodynamics(Lv);
Rv=CavityHemodynamics(Rv);

return
%---- end of main function -----------------------

%==== Auxilary functions ================================

function Seg=VYm2ACXm(Seg);
% input: Vm, Ym,
% output: Am, Cm, Xm

Vm=Seg.Vm; Ym=Seg.Ym;
nt= size(Vm,1);

% Solving 3rd order polynomial
% Mathematica: Solve[x^3 + 3y^2x  - 2V == 0, x]
% Q= (V + Sqrt(V^2 + y^6))^(1/3);  x= Q - y^2/Q;
SignVm= sign(Vm); Vm=abs(Vm);
V     = (3/pi)*Vm;
Q     = (V + sqrt(V.^2 + Ym.^6)).^(1/3);
Xm    = SignVm .* ( Q - Ym.^2 ./ Q );

%calculate midwall area Am and curvature Cm=1/rm
x2    = Xm.^2; y2= Ym.^2;
x2py2 = x2+y2;
Am    = pi*x2py2; % midwall cap area
Cm    = 2*Xm./x2py2; % midwall cap curvature

Seg.Xm = Xm ;
Seg.Am = Am - repmat(Seg.AmDead,[nt,1]);
Seg.Cm = Cm ;

return

function Seg=T2TxTy(Seg);

Ym= Seg.Ym;
Xm= Seg.Xm;
Cm= Seg.Cm;

%calculate midwall area Am and curvature Cm=1/rm
x2=Xm.^2;y2=Ym.^2;
x2py2 = x2+y2;

% Sheet mechanics output
T     = Seg.T;

SinA  = Ym.*Cm;
CosA  = (y2-x2)./x2py2;
Ty    = T.*CosA;
Tx    = T.*SinA;

Seg.Tx    = Tx;
Seg.Ty    = Ty;

return

%====================================================

function Cavity= CavityHemodynamics(Cavity);

rho= 1050;
Cm   = Cavity.Cm;
VWall= Cavity.VWall        ; % wall volume
V    = max(Cavity.V,0)     ; % cavity volume
Len  = 2*(V+VWall).^(1/3)  ; % estimate of long-axis length LV
A    = (V+0.1*VWall) ./Len ; % crude estimate of cross-section of a (left) ventricle

% Estimate of tube properties for wave transmission and valve connections
Cavity.A    = A               ; % cross-sectional area for valve inflow and outflow pressure
Cavity.Z    = 0               ;

% Initializations
Cavity.pIn   = Cavity.p;
Cavity.VDot  = 0;
Cavity.qRemod= 0;
return

%====================================================

function SFn= StructMerge(SnF,StrField);
% SFn= SnF(1); 
SFn=[];
for ifield= 1:length(StrField);
    SFn.(StrField{ifield})= [SnF.(StrField{ifield})];
end
return

