function synthesizer_demo
% synthesizer_demo.m          19/06/2017
% written by C. Hoog Antink
% Copyright: Chair for Medical Information Technology
%            Pauwelsstr. 20
%            52074 Aachen
%
% This code demonstrates the synthesizer described in
%
% C. Hoog Antink, S. Leonhardt, and M. Walter: �A Synthesizer Framework
% for Multimodal Cardiorespiratory Signals�. Biomedical Engineering
% and Physics Express, June 2017
% See in particular Figure 14.
% We kindly ask you to cite the original publication when using this code.

templates = load('templates.mat');

f_HR = 80; % Mean Heart Rate                      [BPM]
f_HF = 15; % Mean Respiratory Frequency           [BPM]
f_LF =  6; % Mean Frequency of Mayer Oscillations [BPM]

k_HF = 0.001;  % Modulation Constant of Respiratory Oscillator
k_LF = 0.0005; % Modulation Constant of Mayer Oscillator

sd_HF = 0.008; % Standard Deviation of Respiratory 'Jitter' Component
sd_LF = 0.008; % Standard Deviation of Mayer 'Jitter' Component

t_sim         = 1000; % Simulation Time              [s]
f_s           = 1000; % Simulated Sampling Frequency [Hz]

t_final  = t_sim * 100;

[b_baro, a_baro] = butter(4,f_LF / 60 / (100 / 2));
[b_RR, a_RR]     = butter(4,f_HF / 60 / (100 / 2));

options = odeset('RelTol',1e-5,'AbsTol',1e-5);

rng(1)
rand_param_LF = filtfilt(b_baro,a_baro,randn(t_final+1,1));
rand_param_HF = filtfilt(b_RR,a_RR,randn(t_final+1,1));
[T,Y] = ode45(@(t,y)rigid(t,y,rand_param_LF,rand_param_HF,f_HR,f_HF,f_LF,sd_LF,sd_HF,k_LF,k_HF),[0 t_final],[1 0 1 0 1 0],options);

delta = 100 / f_s;

interp_equi = (0:delta:T(end))';

sig_2a_equi = interp1(T,Y(:,3),interp_equi);
sig_2b_equi = interp1(T,Y(:,4),interp_equi);
sig_3a_equi = interp1(T,Y(:,5),interp_equi);
sig_3b_equi = interp1(T,Y(:,6),interp_equi);

phi_resp = atan2(sig_2b_equi,sig_2a_equi);
phi_card = atan2(sig_3b_equi,sig_3a_equi);
phi=((1:100) - 1)/100*2*pi-pi;

synt_signal_cell = cell(6,1);

for i=1:6
    card_rec = interp1(phi,templates.T_card_cell{i},phi_card,'spline');
    resp_add_rec = interp1(phi,templates.T_resp_add_cell{i},phi_resp,'spline');
    resp_mod_rec = interp1(phi,templates.T_resp_mod_cell{i},phi_resp,'spline');    
    if i==1
        [~, R] = findpeaks(double(card_rec>2));
    end
    synt_signal_cell{i} = card_rec + resp_add_rec + card_rec .* resp_mod_rec;
end
t = (1:size(card_rec,1))'/f_s;

title_str = {'ECGREF','cECG3','BP','BCG','FLOW1','THORAX'};

h1 = figure;
set(h1,'units','centimeters');
set(h1,'outerposition',[1 1 22 30]);
set(h1,'color','w');
for i=1:6 
    subplot(7,1,i)
    plot(t,synt_signal_cell{i},'k','linewidth',1.5)
    xlim([100 110]);
    x_lim = xlim;
    tmp = synt_signal_cell{i}((t>=x_lim(1))&(t<x_lim(2)));
    ylim([1.05*min(tmp) 1.05*max(tmp)])
    title(title_str{i},'interpreter','latex')    
    ylabel('[a.u.]','interpreter','latex')    
  %  xticklabels({})
end

RR     = diff(R);
RR_loc = R(1:end-1) + 0.5 * RR;
RR     = RR ./ f_s;
RR_loc = RR_loc ./ f_s;

subplot(7,1,7)
plot(RR_loc, RR,'kx','linewidth',1.5)
xlim([100 110]);
x_lim = xlim;
tmp = RR((RR_loc>=x_lim(1))&(RR_loc<x_lim(2)));
ylim([0.95*min(tmp) 1.05*max(tmp)])
title('RR-interval','interpreter','latex')
ylabel('[s]','interpreter','latex')
xlabel('time\,[s]','interpreter','latex')

keyboard

end

function dy = rigid(t,y,rand_param_1,rand_param_2,f_HR,f_HF,f_LF,sd_LF,sd_HF,k_LF,k_HF)
m_1   = rand_param_1(round(t)+1);
m_2   = rand_param_2(round(t)+1);

dy = zeros(2,1);
dy(1) = -2*pi*(f_LF / 60 / 100 + sd_LF*m_1)*y(2);
dy(2) =  2*pi*(f_LF / 60 / 100 + sd_LF*m_1)*y(1);

dy(3) = -2*pi*(f_HF / 60 / 100 + sd_HF*m_2)*y(4);
dy(4) =  2*pi*(f_HF / 60 / 100 + sd_HF*m_2)*y(3);

dy(5) = -2*pi*(f_HR / 60 / 100 + k_LF*y(1)+k_HF*y(3))*y(6);
dy(6) =  2*pi*(f_HR / 60 / 100 + k_LF*y(1)+k_HF*y(3))*y(5);
end